/***************************************************************************
 *
 * Copyright (C) 2001 International Business Machines
 * All rights reserved.
 *
 * This file is part of the GPFS mmfslinux kernel module.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer. 
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written
 *     permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *************************************************************************** */
/*
 * Basic types used throughout GPFS.  Since GPFS shares data between its
 * daemon and its kernel pieces, types must be defined consistently, rather
 * than relying on platform-specific include files.  This file may be
 * included in either C or C++ code.
 *
 * $Id: cxiTypes.h,v 1.55.2.1 2001/10/25 04:55:13 mcnabb Exp $
 *
 * $Log: cxiTypes.h,v $
 * Revision 1.55.2.1  2001/10/25 04:55:13  mcnabb
 * Defect 352112: Be more systematic about kproc startup and end.  Let
 * keventd spawn the kernel threads in linux so we can signal him to
 * reap them.
 *
 * Revision 1.55  2001/10/11 14:04:58  gjertsen
 * Provide support for 64-bit mutex lockword (LOCKWORD64).
 *
 * Revision 1.54  2001/09/29 02:00:38  manoj
 * Fix for Raleigh defect 6854: Ensure that stale pages in Linux's page cache
 * get invalidated on a BR token revoke in both the daemon and kernel versions
 * of flushMappedPages.
 *
 * Revision 1.53  2001/09/24 19:37:48  jpalmer
 * Fix NAS defect 6663.  Replace unused l_vfs lock field in LINUX version with l_caller to distinguish LOCKD requested locks.
 *
 * Revision 1.52  2001/09/22 20:10:26  dcraft
 * Remove kiobufs from cxiKernelIODescriptor_t.  Use temporary
 * kiobufs for map/unmap.   Remove dead code and dead comments
 * in portability layer and update readmes and license.
 * Fix traceback to appear in mmfs.log file.
 *
 * Revision 1.51  2001/09/20 21:46:28  wyllie
 * Revise fsync coalescing
 *
 * Revision 1.50  2001/09/20 05:58:05  wsawdon
 * Defined commands to get/set snapLatest info.
 * Renamed va_xperm to va_xinfo and definef bit flags to allow var to be
 * shared between extended permissions and snapLatest entries.
 *
 * Revision 1.49  2001/09/13 18:35:39  eshel
 * Performance improvement to skip multiple sync calls for the same file.
 *
 * Revision 1.48  2001/09/06 14:00:38  dixonbp
 * Move lock owner from cxiFcntlLock args and add it to cxiFlock_t.
 *
 * Revision 1.47  2001/08/30 23:22:34  yuri
 * On AIX 5.1, ino_t is int32 in 32b mode, and int64 in 64b.  In cxiVattr_t,
 * va_serialno field has to be of the same type as system ino_t since it's
 * typecast to vattr_t on AIX.  GPFS internal inode type must be int32.
 * Create a new type cxiInoSys_t, that corresponds to ino_t on AIX and stays
 * Int32 on Linux, declare va_serialno as cxiInoSys_t.
 *
 * Revision 1.46  2001/08/16 21:23:42  manoj
 * Pass openFile as an argument to breakSMBOplock to check for oplock 
 * status. Move SMB enums from ShHash.h to cxiTypes.h since they are also 
 * used by the portability code.
 *
 * Revision 1.45  2001/08/14 00:06:01  wyllie
 * Do not use C++ style comments in include file that is used from C
 *
 * Revision 1.44  2001/08/13 20:40:52  schmuck
 * Partial fix for Defect 345614 (Raleigh Defect 4423):
 *   The sendfile() system call generates paging requests without an explicit
 * mmap().  When the file is closed, paging requests for read-ahead that were
 * generated as a side effect of sendfile() may still be queued and were not
 * being cleaned up.  A pager kproc would then process these requests using a
 * pointer to the old vinfo structure, which could have been reused for a
 * different file.  This caused inconsistencies in our prefetch buffer
 * accounting, eventually leading to the observed asserts.
 *   Add a flag to the vinfo structure to remember whether we received paging
 * requests for this instance.  On close, check the flag and call mmFlushseg
 * to cancel pending requests if necessary.
 *   Still missing: close needs to wait if a paging request from the instance
 * is currently being processed by a pager kproc.
 *
 * Revision 1.43  2001/08/11 17:22:15  yuri
 * Make cxiPid_t 64-bit on AIX64.  Since pid is 64-bit in kernel and 32-bit in
 * userland, there's no clean way of defining our own pid type, since pid is
 * embedded in many structures in shared segment.  64-bit pid is less painful,
 * however
 *
 * Revision 1.42  2001/07/21 00:30:58  tee
 * Invalidate memory mapped pages when a byte-range write lock is acquired,
 * unless it is being acquired to service a page fault.  This fixes the
 * problem of stale data being seen sometimes by an mmap application even
 * after a sync.  Also, add new option to kxMmapFlush to discard data without
 * flushing.  Use this in trunc to invalidate mapped pages, since flushing is
 * not necessary for pages in the truncated region.
 *
 * Revision 1.41  2001/07/14 00:39:37  eshel
 * Add 'typedef char cBoolean' for a shorter boolean.
 *
 * Revision 1.40  2001/07/05 00:23:46  wyllie
 * Remove support for block trailers
 *
 * Revision 1.39  2001/06/01 18:51:39  gjertsen
 * Introduce macros for intentional conversion between pointers
 * and 32-bit integers so that needless compiler warnings are
 * filtered out while important warnings are now more apparent.
 *
 * Revision 1.38  2001/05/11 20:12:19  eshel
 * Cleanup some GPFS_LINUX ifdefs.
 *
 * Revision 1.37  2001/02/28 17:25:27  wyllie
 * Move definition of EXTERNC to cxiTypes.h
 *
 * Revision 1.36  2001/02/17 02:35:44  wyllie
 * Macros for conversions between big- and little-endian
 *
 * Revision 1.35  2001/02/10 18:13:07  schmuck
 * Tweak ifdef order.
 *
 * Revision 1.34  2000/12/19 16:11:41  wyllie
 * Move Linux mmap declarations out of platform-independent code
 *
 * Replace multiplies, divides, and modulo operations by blocksize by
 * macros that can be configured to either use the native general operators
 * or to use shifts when blocksize is a power of two.
 *
 * Revision 1.33  2000/12/15 13:57:11  gjertsen
 * Clean up documentation.
 *
 * Revision 1.32  2000/12/07 21:38:27  schmuck
 * Add a call to invalidate stat information cached in the Linux inode when
 * the inode token is relinquished/downgraded, so that gpfs_i_revalidate
 * optimization works correctly with multiple nodes.
 * Add similar optimization for gpfs_i_permission.
 * Remove NB_PERF ifdef from this code.
 *
 * Revision 1.31  2000/11/10 03:28:29  radhak
 * More linux mmap code
 *
 * Revision 1.30  2000/11/08 01:12:57  radhak
 * More linux mmap code
 *
 * Revision 1.29  2000/09/15 22:56:02  dcraft
 * Fix sign extension problems on ByteSwap.
 *
 * Revision 1.28  2000/09/01  22:16:25  gjertsen
 * Put in macros that should be used on pointers for pointer arithmetic.
 *
 * Revision 1.27  2000/08/29  18:33:01  dcraft
 * Now produce mmfs module.
 *
 * Revision 1.26  2000/08/29  17:40:06  andreye
 * Add a new platform-independent cxiErrorLogEntry_t type, which describes the
 * contents of a single error log entry. All information stored in the struct of
 * this type in mapped by LogErrors.C onto an err_rec structure in AIX. Linux
 * implementation of the error-logging mechanism requests a format string from
 * the message catalog (based on the error ID field), parses it and constructs a
 * plain text message, formatting the arguments as specified by the format string.
 * The error message gets logged on the system with one or more calls to the
 * standard Linux syslog facility.
 *
 * Revision 1.25  2000/08/24 22:19:21  wyllie
 * Change the typedef for Boolean from 'int' to 'unsigned int'.  Otherwise,
 * on Linux, Booleans that were single bit fields were getting sign-extended.
 * This caused miscompares when comparing a bit field Booleans against 'true',
 * or against other Booleans that were not single bit fields.
 *
 * Revision 1.24  2000/08/21  22:16:01  dcraft
 * Create cxiDev_t type that is based on user level dev_t.  Provide
 * mapping functions between kernel, user, and inode device field.
 * Fix NLS yes/no query.
 *
 * Revision 1.23  2000/08/07  22:33:28  dcraft
 * Use new cxiGetTOD() time of day function for second,
 * nanosecond needs.  Define 32 bit time types and start
 * using in place of current on disk time_t types (necessary
 * for 64 bit).  Use HiResTime where possible.
 *
 * Revision 1.22  2000/08/03  01:06:08  tee
 * Make MIN and MAX declarations match the ones in sys/param.h to avoid
 * redefinition warning.
 *
 * Revision 1.21  2000/08/02 14:56:00  dcraft
 * Typedef Ptrdiff correction from int to long for 64bit
 * Correct quota flags for root user override
 *
 * Revision 1.20  2000/07/31  21:12:50  dcraft
 * Define cxiDirent_t and move some defines out of gpfslinux.h
 * to cxiTypes.h
 *
 * Revision 1.19  2000/07/31  19:04:00  sharma
 * Added Macro  ByteSwap32 and ByteSwap64 to swap 32 bits and 64 bits values
 *
 * Revision 1.18  2000/07/11 16:35:54  wyllie
 * Use cxiUio_t instead of struct uio.  Use cxiUiomove instead of uiomove.  Use
 * CXI_READ instead of UIO_READ, etc.
 *
 * Revision 1.17  2000/06/28  19:23:10  dcraft
 * Remove struct ucred and cred.h dependencies from all OS
 * independent code.  Consistently use cxiUid_t and cxiGid_t.
 * Add linux, aix directories to ts/fs for system dependent code.
 * No-op entry points to DFS export code to avoid ifdefs.
 *
 * Revision 1.16  2000/06/26  16:43:08  dixonbp
 * Create cxiTrace_t to specify kernel trace type for cxiTrace.
 *
 * Revision 1.15  2000/06/23  20:31:37  dcraft
 * Remove sysdep.h and ModeBits.h.  Pare down some OS specific
 * header includes.  Use cxiMode_t instead of mode_t due to
 * linux short mode_t in kernel.
 *
 * Revision 1.14  2000/06/20  23:53:48  dcraft
 * Remove OS specifics from mmfsnode.C and mmfsnode.h. Remove sharkvfs.h.
 * Remove unnecessary OS specific ifdefs of header files.
 *
 * Revision 1.13  2000/06/16  22:21:22  mcnabb
 * Make this includable from non-C++.
 *
 * Revision 1.12  2000/06/16  20:32:57  dcraft
 * Split MMFSNode into cxiNode_t containing OS dependent items
 * (such as gnode) and gpfsNode_t OS independent items.  Acquire
 * inodes during creation of gpfsNodes.  Remove __FULL_PROTO and
 * system prototypes from all C files.  Turn on __FULL_PROTO in
 * site.mcr.
 *
 * Revision 1.11  2000/06/12  21:42:14  dixonbp
 * Move cxiFlock_t from cxiTypes-plat to cxiTypes.
 *
 * Revision 1.10  2000/06/12  19:32:25  dcraft
 * Rename SharkPrivateVFSData to gpfsVFSData_t and split off into OS independent
 * vfsdata.h.   OS dependent layer relies only on forward declaration gpfsVFSData_t
 * since it uses it only as a pointer (hence the non cxi name).
 *
 * Revision 1.9  2000/06/12  18:09:54  wyllie
 * Fix typo.
 *
 * Revision 1.8  2000/06/12  13:20:23  gjertsen
 * Change type names to cxi convention (cxiPid_t and cxiKey_t)
 *
 * Revision 1.7  2000/06/08  13:06:30  gjertsen
 * Introduce cxi types used at user-level and kernel extension
 * interfaces for xmem and uio. Purged out gpfs_flock with cxiFlock_t.
 *
 * Revision 1.6  2000/06/05  19:01:28  dcraft
 * Complete mount callbacks within the OS dependent kernel extension.
 * Record vfs number in gpfsVfsData_t.
 *
 * Revision 1.5  2000/05/31  18:47:10  wyllie
 * Add ifdefs to avoid redefining symbols to reduce the number of warnings
 * from the make Depend step.
 *
 * Revision 1.4  2000/05/30  21:30:07  wyllie
 * Move types from cxiSystemTypes.h to cxiTypes.h.  Use cxi prefix instead of kxi.
 *
 * Revision 1.3  2000/05/30  19:04:58  dcraft
 * Initial drop of vfs ops (mount).  Feng Shui placement of new cxi types.
 *
 * Revision 1.2  2000/05/23  16:19:47  gjertsen
 * Added definitions for common (kernel and daemon) system types.
 * Cleaned up use of pid_t and key_t in code with new types.
 *
 * Revision 1.1  2000/05/19  23:18:47  wyllie
 * Add files to define common (kernel and daemon) types and interfaces.  Move
 * some of the basic types from Shark.h to cxiTypes.h, which is included by
 * Shark.h.
 *
 */

#ifndef _h_cxiTypes
#define _h_cxiTypes

/* Types to use when the number of bits in a word is important.  Use these
   instead of long, short, int, etc. for portability */
typedef short                  Int16;
typedef int                    Int32;
typedef long long              Int64;
typedef unsigned short         UInt16;
typedef unsigned int           UInt32;
typedef unsigned long long     UInt64;

typedef unsigned char          UChar;
typedef unsigned short         UShort;
typedef long                   Ptrdiff;

#define MAX_UINT64 ((UInt64)(0xFFFFFFFFFFFFFFFFuLL))
#define MAX_INT64  ((Int64)(0x7FFFFFFFFFFFFFFFuLL))
#define MAX_UINT32 ((UInt32)0xFFFFFFFF)
#define MAX_INT32  ((Int32)0x7FFFFFFF)
#define MIN_INT32  ((Int32)0x80000000)
#define MAX_UINT16 ((UInt16)0xFFFF)
#define MAX_INT16  ((Int16)0x7FFF)
#define MIN_INT16  ((Int16)0x8000)

/* Macros to fetch the high and low 32 bits of an Int64 */
#define High32(_i) ((Int32) ((_i)>>32))
#define Low32(_i) ((UInt32) ((_i)&0xFFFFFFFF))

/* Use types to safely allow integer to ptr assignment and vice-versa;
   defined for maximum portability (instead of using intptr_t/uintptr_t) */
typedef signed long            IntPtr;
typedef unsigned long          UIntPtr;

/* Define macros to typecast a pointer in a failsafe manner */
#define INTPTR(_p)             ((IntPtr)(_p))
#define UINTPTR(_p)            ((UIntPtr)(_p))

/* Conversion macros for pointers and 32-bit integers */
/* Use when intentionally converting to/from 32-bit integers
   and pointers so that useless compiler warnings are suppressed. */
#define PTR_TO_INT32(_p)         ((Int32)(IntPtr)(_p))
#define PTR_TO_UINT32(_p)        ((UInt32)(UIntPtr)(_p))
#define INT32_TO_PTR(_i)         ((void *)(IntPtr)(_i))
#define UINT32_TO_PTR(_i)        ((void *)(UIntPtr)(_i))

/* Lockword type for mutexes */
#if defined(LOCKWORD64) && !defined(__64BIT__)
#error 64-bit lockword only for 64-bit environment!
#endif /* LOCKWORD64 */
#if defined(LOCKWORD64)
typedef unsigned long          lockWord_t;
#else
typedef unsigned int           lockWord_t;
#endif /* LOCKWORD64 */

/* Boolean type */
typedef unsigned int Boolean;
typedef char cBoolean;
#define false 0
#define FALSE 0
#define true  1
#define TRUE  1

#ifndef EXTERNC
#ifdef __cplusplus
#define EXTERNC extern "C"
#else
#define EXTERNC extern
#endif
#endif /* EXTERNC */

/* Command parameters given to gpfsSetattr() */
#ifndef V_MODE         /* set mode */
#define	V_MODE	0x01
#endif
#ifndef V_OWN          /* set owner */
#define	V_OWN	0x02
#endif
#ifndef V_UTIME        /* set atime and mtime */
#define	V_UTIME	0x04
#endif
#ifndef V_STIME        /* set atime, mtime, and ctime */
#define V_STIME	0x08
#endif
#ifndef V_NTIME        /* set atime, mtime, and ctime to now */
#define V_NTIME 0x10
#endif
#ifndef V_SIZE         /* set file size */
#define V_SIZE  0x20
#endif

#define CXIDEVNONE (cxiDev_t)(-1U)

/* AIX SHM_RDONLY sys/shm.h */
#define CXI_SHM_RDONLY 010000

/* Macros to compute minimum and maximum of two numbers */
#ifndef MIN
#define MIN(a,b) (((a)<(b))?(a):(b))
#endif
#ifndef MAX
#define MAX(a,b) (((a)>(b))?(a):(b))
#endif

/* Make sure null pointer definition is always available */
#ifndef NULL
#define NULL 0
#endif

/* Identifier of a thread.  Always 32 bits, even if native thread IDs are
   longer on some platforms.  The value cxiNoThread denotes an invalid
   thread ID. */
typedef Int32                  cxiThreadId;
#define cxiNoThread            0

#if !defined(GPFS_LINUX) && defined(__64BIT__)
typedef Int64		       cxiPid_t;
#else
typedef Int32                  cxiPid_t;
#endif
typedef Int32                  cxiKey_t;

/* Needed since mode_t is short in linux kernel */
typedef UInt32                 cxiMode_t;
typedef UInt32                 cxiUid_t;
typedef UInt32                 cxiGid_t;
typedef UInt32                 cxiIno_t;    /* gpfs inode number */

/* Original AIX sizes for dev_t.  Special device files continue to
 * record their major/minor numbers as 32 bits on disk.
 */
typedef UInt32                 cxiDev32_t;

typedef UInt32                 cxiTime32_t; /* seconds time type */
typedef UInt32                 cxiNSec32_t; /* nanoseconds time type */
typedef unsigned long          cxiSize_t;

/* File system type within cxiStatfs_t */
typedef struct cxiFsid_t
{
  unsigned long val[2];
} cxiFsid_t;

/* Interface structure for gpfsStatfs() */
typedef struct cxiStatfs_t
{
  int       f_bsize;      /* optimal file system block size */
  int       f_blocks;     /* total data blocks in file system */
  int       f_bfree;      /* free block in fs */
  int       f_bavail;     /* free blocks avail to non-superuser */
  int       f_files;      /* total file nodes in file system */
  int       f_ffree;      /* free file nodes in fs */
  cxiFsid_t f_fsid;       /* file system id */
  int       f_fsize;      /* fundamental file system block size */
  char      f_fname[32];  /* file system name (usually mount pt.) */
  char      f_fpack[32];  /* file system pack name */
  int       f_name_max;   /* maximum component name length for posix */
} cxiStatfs_t;

/* l_caller values */
#define L_CALLER_NULL 0
#define L_CALLER_LOCKD 1

/* Advisory record locking interface structure */
typedef long long cxiOff64_t;
typedef struct cxiFlock_t
{
  short           l_type;
  short           l_whence;
  unsigned int    l_sysid;
  cxiPid_t        l_pid;
#ifndef GPFS_LINUX
  int             l_vfs;
#else
  int             l_caller;
#endif
  cxiOff64_t      l_start;
  cxiOff64_t      l_len;
  void *          l_owner;
} cxiFlock_t;

typedef enum cxiTrace_t
{
  cxiTraceNFS = 0,     /* trace linux nfs activity via printk */
  cxiTraceNFSoff = 1   /* stop  tracing of linux nfs activity */
} cxiTrace_t;


/* DMAPI operation context */
typedef enum cxiContext_t
{
  unknownOp = 0,        /* invalid or unknown operation type */
  vnOp = 1,             /* vnode operation */
  reservedFileOp = 2,   /* quota operation (quota check, asyncRecovery) */
  generalOp = 3,        /* general (not vnop) operation */

  /* must place all non-dm contexts above this line */
  /* must place dmOp context on next line, and all other dm contexts after it */
  dmOp = 4,             /* non-file-related DM operation */
  dmOpWithToken = 5,    /* File-related DM operation with a DM token */
  dmOpWithoutToken = 6, /* File-related DM operation without a token */
  dmOpInvalidToken = 7, /* File-related DM operation with an invalid token */
  dmOpAccessControl = 8 /* DM operation that acquires/releases access rights */
} cxiContext_t;

#define CXI_D_NAME_MAX 255     /* _D_NAME_MAX sys/dir.h */
#define CXI_DEV_BSIZE  512     /* DEV_BSIZE AIX sys/dir.h */
#define CXI_MAXNAMLEN  255     /* MAXNAMLEN AIX sys/dir.h */
#define CXI_MAXPATHLEN 1024    /* MAXPATHLEN AIX sys/param.h */
#define CXI_PATH_MAX   1023    /* PATH_MAX AIX sys/limits.h */

/* This macro verifies if a given operation context is one of DMAPI contexts */
#define IS_DM_CONTEXT(opContext) ((opContext < dmOp) ? false : true)

/* Operations supported by gpfsReadWrite */
typedef enum cxiRdWr_t { CXI_READ, CXI_WRITE, CXI_READ_NO_MOVE,
                         CXI_WRITE_NO_MOVE } cxiRdWr_t;

/* Forward declarations */
struct gpfsVfsData_t;
struct ext_cred_t;

/* Macros to convert integers of various sizes from big endian to little
   endian or vice versa.  Do not use these directly; use either the ByteSwapxx
   routines defined in cxiTypes-plat.h or the BigEndToCPUxx/CPUToBigEndxx
   macros instead. */
#define _ByteSwap16(d) \
        ((UInt16)((((UInt16)(d) >> 8) & 0x00FFU) | \
                  (((UInt16)(d) << 8) & 0xFF00U)))

#define _ByteSwap32(d) \
        ((UInt32)((((UInt32)(d) >> 24) & 0x000000FFU) | \
                  (((UInt32)(d) >>  8) & 0x0000FF00U) | \
                  (((UInt32)(d) <<  8) & 0x00FF0000U) | \
                  (((UInt32)(d) << 24) & 0xFF000000U)))

#define _ByteSwap64(d) \
	((UInt64)((((UInt64)(d) >> 56) & 0x00000000000000FFULL) | \
		  (((UInt64)(d) >> 40) & 0x000000000000FF00ULL) | \
		  (((UInt64)(d) >> 24) & 0x0000000000FF0000ULL) | \
		  (((UInt64)(d) >>  8) & 0x00000000FF000000ULL) | \
		  (((UInt64)(d) <<  8) & 0x000000FF00000000ULL) | \
		  (((UInt64)(d) << 24) & 0x0000FF0000000000ULL) | \
		  (((UInt64)(d) << 40) & 0x00FF000000000000ULL) | \
		  (((UInt64)(d) << 56) & 0xFF00000000000000ULL)))		


/* Include platform specific types */
#include <cxiTypes-plat.h>


/* Macros to be used when switching data from big endian to
 * the endian-ness of the CPU
 */
#ifdef GPFS_LITTLE_ENDIAN
#define BigEndToCPU16(d)  ByteSwap16(d)
#define BigEndToCPU32(d)  ByteSwap32(d)
#define BigEndToCPU64(d)  ByteSwap64(d)
#define CPUToBigEnd16(d)  ByteSwap16(d)
#define CPUToBigEnd32(d)  ByteSwap32(d)
#define CPUToBigEnd64(d)  ByteSwap64(d)
#else
#define BigEndToCPU16(d)  (d)
#define BigEndToCPU32(d)  (d)
#define BigEndToCPU64(d)  (d)
#define CPUToBigEnd16(d)  (d)
#define CPUToBigEnd32(d)  (d)
#define CPUToBigEnd64(d)  (d)
#endif

/* Retrieve the file type from the mode and translate it to cxiNodeType_t */
#define CXIMODE_TO_NODETYPE(M) (modeToNodeType[((M) & S_IFMT) >> 12])

/* Invariant size time structure. */
typedef struct cxiTimeStruc32_t
{
  cxiTime32_t tv_sec;   /* seconds */
  cxiNSec32_t tv_nsec;  /* nanoseconds */
} cxiTimeStruc32_t;

/* May be 32/64 bit specific depending on how OS defines
 * cxiTime_t (cxiTypes-plat.h).  On AIX cxiTime_t (time_t) is
 * an int32long64 and cxiNSec_t (suseconds_t) is a signed int.
 * Nanoseconds can always be represented within 32 bits, however
 * on linux it's a long int, which would change size for 64bits.
 */
typedef struct cxiTimeStruc_t
{
  cxiTime_t tv_sec;    /* seconds */
  cxiNSec_t tv_nsec;   /* nanoseconds */
} cxiTimeStruc_t;

/* Must match struct vattr on AIX */
typedef struct cxiVattr_t
{
  int            va_type;       /* vnode type */
  cxiMode_t      va_mode;       /* access mode */
  cxiUid_t       va_uid;        /* owner uid */
  cxiGid_t       va_gid;        /* owner gid */
  union
  {
    cxiDev_t    _va_dev;        /* id of device containing file	*/
    long        _va_fsid;       /* fs id (dev for now) */
  } _vattr_union;
  cxiInoSys_t    va_serialno;   /* file serial (inode) number */
  short          va_nlink;      /* number of links */
  short          va_flags;      /* Flags, see below for define */
  long           va_mask;       /* Initial attribute mask */
  cxiOff64_t     va_size;       /* file size in bytes */
  long           va_blocksize;  /* preferred blocksize for io */
  long	         va_blocks;     /* kbytes of disk held by file */

  /* The following three fields use the 32/64 bit dependent
   * cxiTimeStruc_t since the AIX vattr structure defines these
   * with timestruc_t which has time_t field.
   */
  cxiTimeStruc_t va_atime;      /* time of last access */
  cxiTimeStruc_t va_mtime;      /* time of last data modification */
  cxiTimeStruc_t va_ctime;      /* time of last status change */
  cxiDev_t       va_rdev;	/* id of device */

  /* Fields added for compatability with the fullstat structure */
  long           va_nid;        /* unused (node id on AIX) */
  int            va_chan;       /* unused (channel on AIX) */
  char          *va_acl;        /* unused (Access Control List on AIX) */
  cxiSize_t      va_xinfo;      /* Extended info field (defined below)
                                   (AIX: cxiSize_t va_aclsiz (size of ACL)) */
  int            va_gen;        /* inode generation number */
} cxiVattr_t;

/* Define bit flags for the extended information field */
#define VA_XPERM 0x0001         // file has extended ACL 
#define VA_XVOL  0x0002         // dir entry was generated and is volatile
 

#ifdef va_dev
#undef va_dev
#endif
#ifdef va_fsid
#undef va_fsid
#endif
#define	va_dev   _vattr_union._va_dev
#define	va_fsid  _vattr_union._va_fsid

/* Based on the AIX definition */
typedef struct cxiIovec_t
{
  char     *iov_base;
  cxiSize_t iov_len;
} cxiIovec_t;

typedef struct cxiErrorLogEntry_t
{
  short logErrorType;       /* One of the values #defined below */

  /*   Fields common for all types of errors   */
  UInt32 logRecTag;         /* Tag to associate related log entries */
  char *componentName;      /* Name of the malfunctioning component */
  char *componentID;
  char *buildBranch;

  /*  Fields used by logMoreInfo, logGenIF, logShutdownIF    */
  char *strFileName;        /* File in which the error occurred */
  UInt32 srcLineNumber;     /* Line on which the error occurred */
  Int32 retCode;            /* return code value */
  Int32 reasonCode;         /* reason code value */
  char *dataStr;            /* Data dump string */
  char *failingExpr;        /* Expression that evaluates to false */

  /*   Fields used by LOGSPECIFIC   */
  UInt32 logTemplID;        /* Error log template ID (used in AIX only) */
  char *formatStr;          /* printf-style format string */
  void *varargPtrPtr;       /* Pointer to the variable argument list
			       (will be casted to va_list *)           */

  char *kernData;           /* Used for copying the logRecData field of the logging
                               kernel mailbox   */
  int kernDataLength;       /* = logRecDataLen */

  char *moduleVersion;      /* Used only for logging from external progsams */
} cxiErrorLogEntry_t;

#ifdef SMB_LOCKS
/* Enumeration used to specify whether a file will be read or written. */
enum AccessMode
{
  opAccessNone  = 0,
  opAccessRead  = 1,
  opAccessWrite = 2
};

/* types of oplocks granted to Samba */
enum OplockMode
{
  smbOplockNone  = 0,
  smbOplockShared  = 1,
  smbOplockExclusive = 2
};

/* Check whether oplock olm is in conflict with lock access am:
   A conflict exists if one of the two is for write/exlucisve and the
   other one is at least for read/shared. */
#define oplockAccessConflict(olm, am) ((int)(olm) + (int)(am) > 2)
#endif

#define LOG_ERR_SPECIFIC    0x1
#define LOG_ERR_GENERIC     0x2
#define LOG_ERR_MOREINFO    0x3
#define LOG_ERR_SHUTDOWN    0x4
#define LOG_ERR_TRACEBACK   0x5
#define LOG_ERR_SPECIFIC_KERN  0x6


/* mmap flush options: */
enum MmflushOption
{
  MmfKeep,          /* Leave pages as they are */
  MmfProtect,       /* Change page protection to read-only */
  MmfInvalidate,    /* Invalidate the pages */
  MmfTerminate,     /* Invalidate the pages and wait for I/O to complete */
  MmfDiscard,       /* Invalidate pages without flushing them */
  MmfCancel,        /* cancel pending requests */
  MmfCacheInval     /* Invalidate the pages in Linux's page cache */
};

/* Common part of information about an open file */
typedef struct cxiVinfo_t
{
  cBoolean inUse;        /* True if VInfo is in use by an open file */
  cBoolean disconnected; /* True if the daemon died or a forced unmount or
                            SG panic occurred since the file was opened */
  cBoolean isNFS;        /* True if file opened for NFS */
  cBoolean rwPageDone;   /* True if paging requests received */
} cxiVinfo_t;

#ifdef _KERNEL
/* Interface structure to cxiStartKProc() and cxiStopKProc() */
typedef struct cxiKProcData_t
{
  cxiPid_t pid;        /* proc pid (returned) */

  /* protects startStopEvent and kprocEvent transitions */
  cxiBlockingMutex_t lock;

  /* kproc can sleep here */
  cxiWaitEvent_t kprocEvent;     

  /* start or stop request should sleep here waiting on pid transition */
  cxiWaitEvent_t startStopEvent; 

  void *kargP;         /* start proc function arguments */
  Boolean terminate;   /* terminate proc when true */
  int (*func)(void*);  /* proc function */
  void *fargP;         /* proc function arguments */
  char *nameP;         /* proc name */
} cxiKProcData_t;

#define KPROC_FAILED_PID      -1
#define KPROC_UNASSIGNED_PID  -2
#define KPROC_RUNNING(KP)     ((KP)->pid > 0)
#endif /* _KERNEL */

/* Macros to do multiplication, division, and modulo by numbers that
   are probably powers of two.  Depending on which define is set, the
   operations can be performed using the built-in operators /, *, and %,
   or using shifts and masks.  The _shift parameter of the macros should
   be log base 2 of _divisor or _multiplier, or 0 if _divisor or
   _multiplier is not a power of 2.  If POW2_COND_SHIFT is defined, the
   generated code will test _shift for 0, then do either a shift or the
   general operation.  If POW2_FORCE_SHIFT is defined, the generated
   code will always use shifts.  This requires that the subblock size of
   the file system actually be a power of 2.  If neither POW2_COND_SHIFT
   nor POW2_FORCE_SHIFT are defined, the generated code will use the
   general operations (/, *, or %). */
#if defined(POW2_FORCE_SHIFT)
# define MULT_OR_SHIFT(_multiplicand, _multiplier, _shift)   \
     ((_multiplicand)<<(_shift))
# define DIV_OR_SHIFT(_dividend, _divisor, _shift)           \
     ((_dividend)>>(_shift))
# define MOD_OR_MASK(_dividend, _divisor, _shift)            \
     ((_dividend) & ((1<<(_shift)) - 1))
#elif defined(POW2_COND_SHIFT)
# define MULT_OR_SHIFT(_multiplicand, _multiplier, _shift)   \
   ( ((_shift)!=0) ?                                         \
       (_multiplicand) << (_shift) :                         \
       ((_multiplicand)*(_multiplier)) )
# define DIV_OR_SHIFT(_dividend, _divisor, _shift)           \
   ( ((_shift)!=0) ?                                         \
       (_dividend) >> (_shift) :                             \
       ((_dividend)/(_divisor)) )
# define MOD_OR_MASK(_dividend, _divisor, _shift)            \
   ( ((_shift)!=0) ?                                         \
       (_dividend) & ((1<<(_shift)) - 1) :                   \
       ((_dividend)%(_divisor)) )
#else
# define MULT_OR_SHIFT(_multiplicand, _multiplier, _shift)   \
     ((_multiplicand)*(_multiplier))
# define DIV_OR_SHIFT(_dividend, _divisor, _shift)           \
     ((_dividend)/(_divisor))
# define MOD_OR_MASK(_dividend, _divisor, _shift)            \
     ((_dividend)%(_divisor))
#endif

/* Flag values for cxiNode_t::icValid:
   Indicates whether inode attributes cached in the osNode (struct inode
   in Linux) are known to be valid. */
#define CXI_IC_PERM   0x00000001  /* permission related inode fields (owner,
                                     mode, acl, etc) are valid */
#define CXI_IC_ATTR   0x00000002  /* other inode fields are valid */

/* combinations of bits defined above: */
#define CXI_IC_STAT   0x00000003  /* what needs to be valid for stat() */
#define CXI_IC_ALL    0x00000003  /* OR of all flags above */

#endif  /* _h_cxiTypes */
